<?php
/**
 * Image.
 *
 * @package Image
 * @category Utils
 */

/**
 * Exit if accessed directly.
 */
defined( 'ABSPATH' ) || exit;

/**
 * Image class.
 */
class CloudFw_Image {
	/**
	 * Image size prefix.
	 *
	 * @var string
	 */
	public static $size_prefix = 'custom_';

	/**
	 * Init method.
	 */
	public static function init() {
		\add_filter( 'image_downsize', [ __CLASS__, 'maybe_resize_image' ], 99, 3 );
	}

	/**
	 * Callback for the "image_downsize" filter.
	 *
	 * @param bool   $retval Ignore this arg.
	 * @param int    $attachment_id Attachment id.
	 * @param string $size_name Image size name.
	 * @return bool
	 */
	public static function maybe_resize_image( $retval = false, $attachment_id = 0, $size_name = null ) {
		if ( ! isset( $size_name ) || ! is_string( $size_name ) || 0 !== strpos( $size_name, self::$size_prefix ) ) {
			return $retval;
		}

		$size_name     = trim( $size_name );
		$attachment_id = (int) $attachment_id;
		$metas         = wp_get_attachment_metadata( $attachment_id, true /*true: Unfiltered*/ );

		if ( empty( $metas ) ) {
			return apply_filters( 'cloudfw_no_image_found', $retval, $size_name, $attachment_id );
		} elseif ( ! isset( $metas['sizes'], $metas['sizes'][ $size_name ] ) ) {
			$sizes = self::detect_size( $size_name );

			if ( ! empty( $sizes['width'] ) || ! empty( $sizes['height'] ) ) {
				$resized_image_data = self::resize_attachment(
					$attachment_id,
					$sizes['width'],
					$sizes['height'],
					$sizes['crop']
				);

				if ( false !== $resized_image_data ) {
					$metas['sizes'][ $size_name ] = [
						'file'      => $resized_image_data['file'],
						'width'     => $resized_image_data['width'],
						'height'    => $resized_image_data['height'],
						'mime-type' => $resized_image_data['mime-type'],
					];

					\wp_update_attachment_metadata( $attachment_id, $metas );

					return [
						$resized_image_data['url'],
						$resized_image_data['width'],
						$resized_image_data['height'],
						true,
					];
				}
			}
		}

		return $retval;
	}

	/**
	 * Detects the image size settings by given string.
	 *
	 * @param string $size_name The size name.
	 * @param bool   $check_for_defaults The check for defaults.
	 * @return array
	 */
	public static function detect_size( $size_name, $check_for_defaults = false ) {
		$width  = 0;
		$height = 0;
		$crop   = false;

		global $_wp_additional_image_sizes;
		if ( isset( $_wp_additional_image_sizes[ $size_name ] ) ) {
			$width  = (int) $_wp_additional_image_sizes[ $size_name ]['width'];
			$height = (int) $_wp_additional_image_sizes[ $size_name ]['height'];
			$crop   = (bool) $_wp_additional_image_sizes[ $size_name ]['crop'];
		} elseif ( preg_match( '/^custom_(\d+)x(\d+)x?(\d*)/', $size_name, $matches ) ) {
			$width  = (int) $matches[1];
			$height = (int) $matches[2];
			$crop   = (bool) $matches[3];
		} elseif ( $check_for_defaults ) {
			$width  = (int) get_option( "{$size_name}_size_w" );
			$height = (int) get_option( "{$size_name}_size_h" );
			$crop   = (bool) get_option( "{$size_name}_crop" );
		}

		return compact( 'width', 'height', 'crop' );
	}

	/**
	 * Creates a resized/cropped version of an image by a given attachment ID.
	 *
	 * @param int  $attachment_id Attachment id.
	 * @param int  $width Image width.
	 * @param int  $height Image height.
	 * @param bool $crop Crop image.
	 * @return string|false
	 */
	private static function resize_attachment( $attachment_id = 0, $width = 0, $height = 0, $crop = true ) {
		$attachment_id = (int) $attachment_id;
		$width         = (int) $width;
		$height        = (int) $height;
		$crop          = (bool) $crop;

		$attachment = get_post( $attachment_id );

		if ( ! $attachment || 'attachment' !== $attachment->post_type || 'image/' !== substr( $attachment->post_mime_type, 0, 6 ) ) {
			return false;
		}

		if ( ! function_exists( '\wp_get_image_editor' ) ) {
			include_once ABSPATH . 'wp-admin/includes/media.php';
		}

		// Ger image file path.
		$original_image_file_path = get_attached_file( $attachment->ID );

		$image = wp_get_image_editor( $original_image_file_path );

		if ( is_wp_error( $image ) ) {
			return false;
		}

		if ( 0 === $width ) {
			$width = null;
		}

		if ( 0 === $height ) {
			$height = null;
		}

		$resized = $image->resize( $width, $height, $crop );

		if ( is_wp_error( $resized ) ) {
			if ( ! in_array( 'error_getting_dimensions', $resized->get_error_codes(), true ) ) {
				return false;
			}

			// Return original size info.
			$sizes = $image->get_size();
			return [
				'url'       => $attachment->guid,
				'file'      => wp_basename( $original_image_file_path ),
				'width'     => $sizes['width'],
				'height'    => $sizes['height'],
				'mime-type' => $attachment->post_mime_type,
			];
		} else {
			// Save image.
			$resized_image_data = $image->save();

			if ( ! is_wp_error( $resized_image_data ) ) {
				$wp_uploads     = wp_get_upload_dir();
				$wp_uploads_dir = $wp_uploads['basedir'];
				$wp_uploads_url = $wp_uploads['baseurl'];

				return [
					'url'       => str_replace( $wp_uploads_dir, $wp_uploads_url, $resized_image_data['path'] ),
					'file'      => $resized_image_data['file'],
					'width'     => $resized_image_data['width'],
					'height'    => $resized_image_data['height'],
					'mime-type' => $resized_image_data['mime-type'],
				];
			}
		}

		return false;
	}
}
